/*
 * Decompiled with CFR 0.152.
 */
package reactor.core.scheduler;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Supplier;
import reactor.core.Cancellation;
import reactor.core.Exceptions;
import reactor.core.scheduler.ElasticScheduler;
import reactor.core.scheduler.ExecutorScheduler;
import reactor.core.scheduler.ExecutorServiceScheduler;
import reactor.core.scheduler.ImmediateScheduler;
import reactor.core.scheduler.ParallelScheduler;
import reactor.core.scheduler.Scheduler;
import reactor.core.scheduler.SingleScheduler;
import reactor.core.scheduler.SingleTimedScheduler;
import reactor.core.scheduler.SingleWorkerScheduler;
import reactor.core.scheduler.TimedScheduler;
import reactor.util.Logger;
import reactor.util.Loggers;

public class Schedulers {
    public static final int DEFAULT_POOL_SIZE = Math.max(Runtime.getRuntime().availableProcessors(), 4);
    static final String ELASTIC = "elastic";
    static final String PARALLEL = "parallel";
    static final String SINGLE = "single";
    static final String TIMER = "timer";
    static final ConcurrentMap<String, CachedScheduler> cachedSchedulers = new ConcurrentHashMap<String, CachedScheduler>();
    static final Supplier<Scheduler> ELASTIC_SUPPLIER = () -> Schedulers.newElastic(ELASTIC, 60, true);
    static final Supplier<Scheduler> PARALLEL_SUPPLIER = () -> Schedulers.newParallel(PARALLEL, Runtime.getRuntime().availableProcessors(), true);
    static final Supplier<Scheduler> SINGLE_SUPPLIER = () -> Schedulers.newSingle(SINGLE, true);
    static final Supplier<TimedScheduler> TIMER_SUPPLIER = () -> Schedulers.newTimer(TIMER);
    static final Factory DEFAULT;
    static volatile Factory factory;
    static final Logger log;

    public static Scheduler fromExecutor(Executor executor) {
        return Schedulers.fromExecutor(executor, false);
    }

    public static Scheduler fromExecutor(Executor executor, boolean trampoline) {
        return new ExecutorScheduler(executor, trampoline);
    }

    public static Scheduler fromExecutorService(ExecutorService executorService) {
        return Schedulers.fromExecutorService(executorService, false);
    }

    public static Scheduler fromExecutorService(ExecutorService executorService, boolean interruptOnCancel) {
        return new ExecutorServiceScheduler(executorService, interruptOnCancel);
    }

    public static Scheduler elastic() {
        return Schedulers.cache(ELASTIC, ELASTIC_SUPPLIER);
    }

    public static Scheduler immediate() {
        return ImmediateScheduler.instance();
    }

    public static Scheduler newElastic(String name) {
        return Schedulers.newElastic(name, 60);
    }

    public static Scheduler newElastic(String name, int ttlSeconds) {
        return Schedulers.newElastic(name, ttlSeconds, false);
    }

    public static Scheduler newElastic(String name, int ttlSeconds, boolean daemon) {
        return Schedulers.newElastic(ttlSeconds, new SchedulerThreadFactory(name, daemon, ElasticScheduler.COUNTER));
    }

    public static Scheduler newElastic(int ttlSeconds, ThreadFactory threadFactory) {
        return factory.newElastic(ttlSeconds, threadFactory);
    }

    public static Scheduler newParallel(String name) {
        return Schedulers.newParallel(name, Runtime.getRuntime().availableProcessors());
    }

    public static Scheduler newParallel(String name, int parallelism) {
        return Schedulers.newParallel(name, parallelism, false);
    }

    public static Scheduler newParallel(String name, int parallelism, boolean daemon) {
        return Schedulers.newParallel(parallelism, new SchedulerThreadFactory(name, daemon, ParallelScheduler.COUNTER));
    }

    public static Scheduler newParallel(int parallelism, ThreadFactory threadFactory) {
        return factory.newParallel(parallelism, threadFactory);
    }

    public static Scheduler newSingle(String name) {
        return Schedulers.newSingle(name, false);
    }

    public static Scheduler newSingle(String name, boolean daemon) {
        return Schedulers.newSingle(new SchedulerThreadFactory(name, daemon, SingleScheduler.COUNTER));
    }

    public static Scheduler newSingle(ThreadFactory threadFactory) {
        return factory.newSingle(threadFactory);
    }

    public static TimedScheduler newTimer(String name) {
        return Schedulers.newTimer(name, true);
    }

    public static TimedScheduler newTimer(String name, boolean daemon) {
        return Schedulers.newTimer(new SchedulerThreadFactory(name, daemon, SingleTimedScheduler.COUNTER));
    }

    public static TimedScheduler newTimer(ThreadFactory threadFactory) {
        return factory.newTimer(threadFactory);
    }

    public static Scheduler parallel() {
        return Schedulers.cache(PARALLEL, PARALLEL_SUPPLIER);
    }

    public static void resetFactory() {
        Schedulers.setFactory(DEFAULT);
    }

    public static void setFactory(Factory factoryInstance) {
        Objects.requireNonNull(factoryInstance, "factoryInstance");
        Schedulers.shutdownNow();
        factory = factoryInstance;
    }

    public static void shutdownNow() {
        Collection view = cachedSchedulers.values();
        do {
            ArrayList schedulers = new ArrayList(view);
            view.clear();
            schedulers.forEach(CachedScheduler::_shutdown);
        } while (!view.isEmpty());
    }

    public static Scheduler single() {
        return Schedulers.cache(SINGLE, SINGLE_SUPPLIER);
    }

    public static Scheduler single(Scheduler original) {
        return new SingleWorkerScheduler(original);
    }

    public static TimedScheduler timer() {
        return Schedulers.timedCache(TIMER, TIMER_SUPPLIER).asTimedScheduler();
    }

    static CachedScheduler cache(String key, Supplier<Scheduler> schedulerSupplier) {
        CachedScheduler s;
        while ((s = (CachedScheduler)cachedSchedulers.get(key)) == null) {
            s = new CachedScheduler(key, schedulerSupplier.get());
            if (cachedSchedulers.putIfAbsent(key, s) == null) {
                return s;
            }
            s._shutdown();
        }
        return s;
    }

    static CachedScheduler timedCache(String key, Supplier<TimedScheduler> schedulerSupplier) {
        CachedScheduler s;
        while ((s = (CachedScheduler)cachedSchedulers.get(key)) == null) {
            s = new CachedTimedScheduler(key, schedulerSupplier.get());
            if (cachedSchedulers.putIfAbsent(key, s) == null) {
                return s;
            }
            s._shutdown();
        }
        return s;
    }

    static void handleError(Throwable ex) {
        Throwable t = Exceptions.unwrap(ex);
        Exceptions.throwIfFatal(t);
        Thread.UncaughtExceptionHandler x = Thread.currentThread().getUncaughtExceptionHandler();
        if (x != null) {
            x.uncaughtException(Thread.currentThread(), t);
        } else {
            log.error("Scheduler worker failed with an uncaught exception", t);
        }
    }

    static {
        factory = DEFAULT = new Factory(){};
        log = Loggers.getLogger(Schedulers.class);
    }

    static final class CachedTimedScheduler
    extends CachedScheduler
    implements TimedScheduler {
        final TimedScheduler cachedTimed;

        CachedTimedScheduler(String key, TimedScheduler cachedTimed) {
            super(key, cachedTimed);
            this.cachedTimed = cachedTimed;
        }

        @Override
        public Cancellation schedule(Runnable task, long delay, TimeUnit unit) {
            return this.cachedTimed.schedule(task, delay, unit);
        }

        @Override
        public Cancellation schedulePeriodically(Runnable task, long initialDelay, long period, TimeUnit unit) {
            return this.cachedTimed.schedulePeriodically(task, initialDelay, period, unit);
        }

        @Override
        public TimedScheduler.TimedWorker createWorker() {
            return this.cachedTimed.createWorker();
        }

        @Override
        TimedScheduler asTimedScheduler() {
            return this;
        }

        @Override
        public TimedScheduler get() {
            return this.cachedTimed;
        }
    }

    static class CachedScheduler
    implements Scheduler,
    Supplier<Scheduler> {
        final Scheduler cached;
        final String key;

        CachedScheduler(String key, Scheduler cached) {
            this.cached = cached;
            this.key = key;
        }

        @Override
        public Cancellation schedule(Runnable task) {
            return this.cached.schedule(task);
        }

        @Override
        public Scheduler.Worker createWorker() {
            return this.cached.createWorker();
        }

        @Override
        public void start() {
            this.cached.start();
        }

        @Override
        public void shutdown() {
        }

        @Override
        public Scheduler get() {
            return this.cached;
        }

        void _shutdown() {
            this.cached.shutdown();
        }

        TimedScheduler asTimedScheduler() {
            throw new UnsupportedOperationException("Scheduler is not Timed");
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            CachedScheduler scheduler = (CachedScheduler)o;
            return this.cached.equals(scheduler.cached);
        }

        public int hashCode() {
            return this.cached.hashCode();
        }
    }

    static final class SchedulerThreadFactory
    implements ThreadFactory,
    Supplier<String>,
    Thread.UncaughtExceptionHandler {
        final String name;
        final boolean daemon;
        final AtomicLong COUNTER;

        SchedulerThreadFactory(String name, boolean daemon, AtomicLong counter) {
            this.name = name;
            this.daemon = daemon;
            this.COUNTER = counter;
        }

        @Override
        public Thread newThread(Runnable r) {
            Thread t = new Thread(r, this.name + "-" + this.COUNTER.incrementAndGet());
            t.setDaemon(this.daemon);
            t.setUncaughtExceptionHandler(this);
            return t;
        }

        @Override
        public void uncaughtException(Thread t, Throwable e) {
            log.error("Scheduler worker failed with an uncaught exception", e);
        }

        @Override
        public String get() {
            return this.name;
        }
    }

    public static interface Factory {
        default public Scheduler newElastic(int ttlSeconds, ThreadFactory threadFactory) {
            return new ElasticScheduler(threadFactory, ttlSeconds);
        }

        default public Scheduler newParallel(int parallelism, ThreadFactory threadFactory) {
            return new ParallelScheduler(parallelism, threadFactory);
        }

        default public Scheduler newSingle(ThreadFactory threadFactory) {
            return new SingleScheduler(threadFactory);
        }

        default public TimedScheduler newTimer(ThreadFactory threadFactory) {
            return new SingleTimedScheduler(threadFactory);
        }
    }
}

