/*
 * JBoss, Home of Professional Open Source.
 *
 * See the LEGAL.txt file distributed with this work for information regarding copyright ownership and licensing.
 *
 * See the AUTHORS.txt file distributed with this work for a full listing of individual contributors.
 */
package org.teiid.designer.mapping.ui.actions;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import org.eclipse.core.resources.IResource;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.PartInitException;
import org.teiid.designer.core.workspace.ModelResource;
import org.teiid.designer.mapping.factory.ModelMapperFactory;
import org.teiid.designer.mapping.ui.UiConstants;
import org.teiid.designer.mapping.ui.UiPlugin;
import org.teiid.designer.ui.common.eventsupport.SelectionUtilities;
import org.teiid.designer.ui.common.product.ProductCustomizerMgr;
import org.teiid.designer.ui.common.util.UiUtil;
import org.teiid.designer.ui.common.util.WidgetUtil;
import org.teiid.designer.ui.viewsupport.ModelUtilities;



/** 
 * @since 8.0
 */
public class FindXsdComponentAction extends MappingAction {
    
    //============================================================================================================================
    // Constants
    
//    private static final String PREFIX = I18nUtil.getPropertyPrefix(FindXsdComponentAction.class);

    //============================================================================================================================
    // Constructors
    
    /**
     * Construct an instance of GenerateDependencyReportAction.
     * 
     */
    public FindXsdComponentAction() {
        super();
        this.setUseWaitCursor(false);
        setImageDescriptor(UiPlugin.getDefault().getImageDescriptor(UiConstants.Images.FIND_XSD_COMPONENT));
    }
    
    //============================================================================================================================
    // ISelectionListener Methods
    
    /**
     * @see org.eclipse.ui.ISelectionListener#selectionChanged(IWorkbenchPart, ISelection)
     * @since 4.0
     */
    @Override
    public void selectionChanged(final IWorkbenchPart part, final ISelection selection) {
        super.selectionChanged(part, selection);
        determineEnablement();
    }

    //============================================================================================================================
    // Action Methods

    /**
     * @see org.eclipse.jface.action.Action#run()
     * @since 4.0
     */
    @Override
    protected void doRun() {
        final EObject eObject = SelectionUtilities.getSelectedEObject(getSelection());
        if( eObject != null && ModelMapperFactory.isXmlTreeNode(eObject) ) {
            // Get the xsd component
            final EObject xsdComponent = ModelMapperFactory.getXsdComponent(eObject);
            if( xsdComponent != null ) {
                Display.getCurrent().asyncExec(new Runnable() {
                    @Override
					public void run() {
                        try {
                            final ISelection selection = new StructuredSelection(xsdComponent);
                            String viewId = ProductCustomizerMgr.getInstance().getProductCharacteristics().getPrimaryNavigationViewId();
                            
                            IViewPart view = UiUtil.getWorkbenchPage().showView(viewId);
                            ISelectionProvider selProvider = view.getViewSite().getSelectionProvider();
                            if( selProvider != null ) {
                                if( selProvider instanceof TreeViewer ) { 
                                    showEObject((TreeViewer)selProvider, xsdComponent);
                                }
                                selProvider.setSelection(selection);
//                                view.getTreeViewer().setSelection(selection);
                            }
                        } catch (PartInitException err) {
                            UiConstants.Util.log(err);
                            WidgetUtil.showError(err.getLocalizedMessage());
                        }
                    }
                });
            }
        }
        determineEnablement();
    }
    
    //============================================================================================================================
    // Declared Methods
    
    /**
     * @since 4.0
     */
    private void determineEnablement() {
        boolean enable = false;
        final EObject eObject = SelectionUtilities.getSelectedEObject(getSelection());
        if( eObject != null && ModelMapperFactory.isXmlTreeNode(eObject)) {
            enable = true;
        }

        setEnabled(enable);
    }
    
    /** Gets the ancestors of the node in the Project and Model hierarcy
     *  in the order specified
     * 
     * @param node The node to work with
     * @param oldestFirst if true, order the list from "oldest" (most distant) ancestor to nearest ancestor 
     *  (immediate parent).  If false, order from nearest to oldest.
     * @return
     */
   public List getModelExplorerAncestors(EObject node, boolean oldestFirst) {
       List rv = new ArrayList();

       EObject parentNode = node.eContainer();

       while (parentNode != null) {
           rv.add(parentNode);
           parentNode = parentNode.eContainer();
       } // endwhile

       ModelResource mr = ModelUtilities.getModelResourceForModelObject(node);
       IResource res = mr.getResource();
       rv.add(res);
       IResource parentRes = res.getParent();
       while (parentRes != null) {
           rv.add(parentRes);
           parentRes = parentRes.getParent();
       } // endwhile

       if (oldestFirst) {
           Collections.reverse(rv);
       } // endif

       return rv;
   }

   public void showEObject(TreeViewer viewer, EObject nodeToShow) {
       List l = getModelExplorerAncestors(nodeToShow, true);
       
       Iterator itor = l.iterator();
       while (itor.hasNext()) {
           Object parentNode = itor.next();
           viewer.setExpandedState(parentNode, true);
       } // endwhile
   }
    
}
