/*
 * JBoss, Home of Professional Open Source.
 *
 * See the LEGAL.txt file distributed with this work for information regarding copyright ownership and licensing.
 *
 * See the AUTHORS.txt file distributed with this work for a full listing of individual contributors.
 */
package org.teiid.designer.mapping.ui.actions;

import java.util.Iterator;
import java.util.List;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.emf.ecore.EObject;
import org.teiid.designer.core.workspace.ModelResource;
import org.teiid.designer.core.workspace.ModelWorkspaceException;
import org.teiid.designer.diagram.ui.DiagramUiConstants;
import org.teiid.designer.diagram.ui.DiagramUiPlugin;
import org.teiid.designer.diagram.ui.actions.DiagramEditorAction;
import org.teiid.designer.diagram.ui.model.DiagramModelNode;
import org.teiid.designer.mapping.ui.PluginConstants;
import org.teiid.designer.mapping.ui.UiConstants;
import org.teiid.designer.mapping.ui.diagram.MappingDiagramUtil;
import org.teiid.designer.metamodels.diagram.Diagram;
import org.teiid.designer.ui.common.util.UiUtil;
import org.teiid.designer.ui.editors.ModelEditorManager;
import org.teiid.designer.ui.viewsupport.ModelUtilities;


/**
 * ShowParentDiagramAction
 *
 * @since 8.0
 */
public class ShowParentDiagramAction extends DiagramEditorAction {
    private static final String textString = "org.teiid.designer.mapping.ui.actions.ShowParentDiagramAction.text";  //$NON-NLS-1$
    private static final String toolTipString = "org.teiid.designer.mapping.ui.actions.ShowParentDiagramAction.toolTip";  //$NON-NLS-1$
    /**
     * Construct an instance of ShowParentDiagramAction.
     * 
     */
    public ShowParentDiagramAction() {
        super();
        setImageDescriptor(DiagramUiPlugin.getDefault().getImageDescriptor(DiagramUiConstants.Images.UP_PACKAGE_DIAGRAM));
        setToolTipText(UiConstants.Util.getString(toolTipString));
        setText(UiConstants.Util.getString(textString));
        setEnabled(false);
    }

    /* (non-Javadoc)
     * see org.teiid.designer.ui.common.actions.AbstractAction#doRun()
     */
    @Override
    protected void doRun() {
        if( editor != null ) {
            // CLose object editor
            ModelEditorManager.closeObjectEditor();
            
            Diagram diagram = getParentDiagram();
            if( diagram != null ) {
                // Mark current navigation location using current open object
                UiUtil.getWorkbenchPage().getNavigationHistory().markLocation(editor);
                ModelEditorManager.closeObjectEditor();
                editor.openContext(diagram);
            }
        }
        determineEnablement();
    }
    
    public void determineEnablement() {
        if( editor != null ) {
            Diagram diagram = getParentDiagram();
            if( diagram != null )
                setEnabled(true);
            else
                setEnabled(false);
        }

    }
    
    private Diagram getParentDiagram() {
        Diagram parentDiagram = null;
        
        DiagramModelNode diagramNode = editor.getCurrentModel();
        
        if( diagramNode != null ) {
            Diagram currentDiagram = (Diagram)diagramNode.getModelObject();
            
            if( currentDiagram != null &&
                currentDiagram.getType() != null ) {
                ModelResource modelResource = ModelUtilities.getModelResourceForModelObject(currentDiagram);
                if( modelResource != null ) {
                    if( currentDiagram.getType().equals(PluginConstants.MAPPING_DIAGRAM_TYPE_ID) ) {
                        Object diagramTarget = currentDiagram.getTarget();
                        if( diagramTarget != null && 
                            diagramTarget instanceof EObject ) {
                              // This is a package under a model, so..... get the diagram for the resource?
                              try {
                                  // get diagrams and find package diagram.
                                  List diagramList = modelResource.getModelDiagrams().getDiagrams(null);
                                  Iterator iter = diagramList.iterator();
                                  Diagram nextDiagram = null;
                                  while( iter.hasNext() && parentDiagram == null ) {
                                      nextDiagram = (Diagram)iter.next();
                                      if( nextDiagram.getType() != null 
                                       && nextDiagram.getType().equals(org.teiid.designer.diagram.ui.PluginConstants.PACKAGE_DIAGRAM_TYPE_ID) ) {
                                          parentDiagram = nextDiagram;
                                      } // endif
                                  } // endwhile

                                  // defect 16988 - create the parent if it doesn't exist yet:
                                  if (parentDiagram == null) {
                                      // we need to create the package diagram, since it has not existed before.
                                      // Doing this will make the model dirty.
                                      parentDiagram = DiagramUiPlugin.getDiagramTypeManager().getDiagramForContext(modelResource);
                                  } // endif
                              } catch (ModelWorkspaceException e) {
                                  String message = "ShowParentDiagramAction cannot find parent package diagram";  //$NON-NLS-1$
                                  DiagramUiConstants.Util.log(IStatus.ERROR, e, message);
                              }
                        }
                    } else if( currentDiagram.getType().equals(PluginConstants.MAPPING_TRANSFORMATION_DIAGRAM_TYPE_ID) ) {
                        parentDiagram = MappingDiagramUtil.getCoarseMappingDiagram(currentDiagram);
                    }
                }
            }
        }
        
        return parentDiagram;
    }
    
}
