/*
 * JBoss, Home of Professional Open Source.
 *
 * See the LEGAL.txt file distributed with this work for information regarding copyright ownership and licensing.
 *
 * See the AUTHORS.txt file distributed with this work for a full listing of individual contributors.
 */
package org.teiid.designer.core.metamodel.aspect.sql;

import java.util.List;

import org.eclipse.emf.ecore.EObject;

/**
 * SqlProcedureAspect
 *
 * @since 8.0
 */
public interface SqlProcedureAspect extends
                                   SqlAspect {

    // These are mapping types for which the procedure can be atarget.
    interface MAPPINGS {

        public static final int SQL_TRANSFORM = 1;
    }

    /**
     * Check if table represents a procedure in a virtual model
     * 
     * @return true if the procedure is virtual
     */
    boolean isVirtual(EObject eObject);

    /**
     * Check if this aspect represents a procedure that is a funtion.
     * 
     * @param eObject
     *            The <code>EObject</code> for procedure object
     * @return true if the procedure is a function, else false.
     */
    boolean isFunction(EObject eObject);

    /**
     * Get a list of parameters for the given procedure
     * 
     * @param eObject
     *            The <code>EObject</code> for procedure object
     * @return a list of parameters on the procedure
     */
    List getParameters(EObject eObject);

    /**
     * Get the resultSet on a given procedure.
     * 
     * @param eObject
     *            The <code>EObject</code> for procedure object
     * @return The ResultSet on the procedure
     */
    Object getResult(EObject eObject);

    /**
     * Returns an integer value indicating the number of updates
     * the user has specified will occur within the supplied procedure.
     * 
     * @param eObject
     *            A physical stored procedure.
     * @return The update count value.
     * @since 5.5.3
     */
    int getUpdateCount(EObject eObject);

    /**
     * Check if the procedure is mappable for the given mapping type.
     * 
     * @param eObject
     *            The <code>EObject</code> whose mapability is checked
     * @param mappinType
     *            The mapping type being checked.
     * @return true if mappable else false
     */
    boolean isMappable(EObject eObject,
                       int mappingType);

    /**
     * Check if the procedure is can be a transformation source for the given target.
     * 
     * @param source
     *            The <code>EObject</code> intends to be a transformation source
     * @param target
     *            The <code>EObject</code> that is the target of the transformation
     * @return true if can be a source else false
     */
    boolean canBeTransformationSource(EObject source,
                                      EObject target);

    /**
     * Check if the procedure can accept the given transformation source.
     * 
     * @param target
     *            The <code>EObject</code> that is the target of the transformation
     * @param source
     *            The <code>EObject</code> intends to be a transformation source
     * @return true if can be a source else false
     */
    boolean canAcceptTransformationSource(EObject target,
                                          EObject source);
}
