/*
 * JBoss, Home of Professional Open Source.
 *
 * See the LEGAL.txt file distributed with this work for information regarding copyright ownership and licensing.
 *
 * See the AUTHORS.txt file distributed with this work for a full listing of individual contributors.
 */
package org.teiid.designer.runtime.ui.connection;

import static org.teiid.designer.runtime.DqpPlugin.Util;
import static org.teiid.designer.runtime.ui.DqpUiConstants.UTIL;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Properties;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IAction;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.widgets.Display;
import org.teiid.core.designer.properties.PropertyDefinition;
import org.teiid.core.designer.util.CoreArgCheck;
import org.teiid.core.designer.util.CoreStringUtil;
import org.teiid.core.designer.util.I18nUtil;
import org.teiid.core.designer.util.StringUtilities;
import org.teiid.designer.core.ModelerCore;
import org.teiid.designer.core.workspace.ModelWorkspaceException;
import org.teiid.designer.metamodels.core.ModelType;
import org.teiid.designer.runtime.DqpPlugin;
import org.teiid.designer.runtime.connection.ModelConnectionMapper;
import org.teiid.designer.runtime.connection.TranslatorUtils;
import org.teiid.designer.runtime.spi.ITeiidDataSource;
import org.teiid.designer.runtime.spi.ITeiidServer;
import org.teiid.designer.runtime.spi.ITeiidTranslator;
import org.teiid.designer.vdb.VdbModelEntry;
import org.teiid.designer.vdb.connections.SourceHandler;
import org.teiid.designer.vdb.connections.VdbSourceConnection;


/**
 * Implements the SourceHandler interface which provides the VDB Editor the ability to access DQP-related connection info.
 * 
 *
 * @since 8.0
 */
public class VdbSourceConnectionHandler implements SourceHandler {
    static final String PREFIX = I18nUtil.getPropertyPrefix(VdbSourceConnectionHandler.class);

    private static final String JNDI_PROPERTY_KEY = "jndi-name"; //$NON-NLS-1$
    
    private static SelectTranslatorAction selectTranslatorAction;

    private static SelectJndiDataSourceAction selectJndiDataSourceAction;

    private static Object[] actions;

    private static boolean initialized = false;

    static String getString( final String stringId ) {
        return UTIL.getString(PREFIX + stringId);
    }

    @Override
    public VdbSourceConnection ensureVdbSourceConnection( String sourceModelname,
                                                          Properties properties ) throws Exception {
        CoreArgCheck.isNotNull(properties, "properties"); //$NON-NLS-1$

        ModelConnectionMapper mapper = new ModelConnectionMapper(sourceModelname, properties);

        VdbSourceConnection vdbSourceConnection = null;

        String uuid = ModelerCore.workspaceUuid().toString();

        try {
            vdbSourceConnection = mapper.getVdbSourceConnection(getDefaultServer(), uuid);
        } catch (ModelWorkspaceException e) {
            UTIL.log(IStatus.ERROR,
                     e,
                     UTIL.getString("VdbSourceConnectionHandler.Error_could_not_find_source_connection_info_for_{0}_model", sourceModelname)); //$NON-NLS-1$
        }

        // TODO: vdbSourceConnection may be NULL, so query the user for translator name & jndi name

        return vdbSourceConnection;
    }

    @Override
    public Object[] getApplicableActions( Object obj ) {
        if (!initialized) {
            initialize();
        }
        ITeiidServer defServer = getDefaultServer();
        if (defServer == null || !defServer.isConnected()) {
            return null;
        }

        if (obj instanceof IStructuredSelection) {
            IStructuredSelection sel = (IStructuredSelection)obj;
            if (sel.getFirstElement() instanceof VdbModelEntry) {
            	if (((VdbModelEntry)sel.getFirstElement()).getType() == ModelType.PHYSICAL_LITERAL.getName()) {
                    selectTranslatorAction.setSelection((VdbModelEntry)sel.getFirstElement());
                    selectJndiDataSourceAction.setSelection((VdbModelEntry)sel.getFirstElement());
                    return actions;
                }
            }
        }
        selectTranslatorAction.setSelection(null);
        selectJndiDataSourceAction.setSelection(null);
        return null;
    }

    private void initialize() {
        // Construct the two actions
        selectTranslatorAction = new SelectTranslatorAction(getString("selectTranslatorAction.label")); //$NON-NLS-1$

        selectJndiDataSourceAction = new SelectJndiDataSourceAction(getString("selectJndiDataSourceAction.label")); //$NON-NLS-1$

        Collection<IAction> actionsList = new ArrayList<IAction>();
        actionsList.add(selectTranslatorAction);
        actionsList.add(selectJndiDataSourceAction);

        actions = actionsList.toArray();
    }

    class SelectJndiDataSourceAction extends Action {

        public SelectJndiDataSourceAction( String text ) {
            super(text);
            // TODO Auto-generated constructor stub
        }

        private VdbModelEntry vdbModelEntry;

        public void setSelection( VdbModelEntry vdbModelEntry ) {
            this.vdbModelEntry = vdbModelEntry;
        }

        @Override
        public void run() {
            // Get available servers and launch SelectTranslatorDialog
            // vdbModelEntry should not be null and should be a Physical model only
            if (vdbModelEntry != null) {
                String jndiName = vdbModelEntry.getSourceInfo().getSource(0).getJndiName();

                SelectJndiDataSourceDialog dialog = new SelectJndiDataSourceDialog(Display.getCurrent().getActiveShell());

                ITeiidDataSource initialSelection = null;
                ITeiidServer defServer = getDefaultServer();
                if (defServer != null && defServer.isConnected()) {
                    try {
                        initialSelection = defServer.getDataSource(jndiName);
                    } catch (Exception e) {
                        UTIL.log(IStatus.ERROR,
                                 e,
                                 UTIL.getString("VdbSourceConnectionHandler.Error_could_not_find_data_source_for_name", jndiName)); //$NON-NLS-1$
                    }
                    dialog.setInitialSelection(initialSelection);
                }

                dialog.open();

                if (dialog.getReturnCode() == Window.OK) {
                    Object result = dialog.getFirstResult();
                    if (result != null && result instanceof ITeiidDataSource) {
                        vdbModelEntry.setJndiName(0, ((ITeiidDataSource)result).getName());
                    }
                }
            }
        }
    }

    class SelectTranslatorAction extends Action {

        public SelectTranslatorAction( String text ) {
            super(text);
            // TODO Auto-generated constructor stub
        }

        private VdbModelEntry vdbModelEntry;

        public void setSelection( VdbModelEntry vdbModelEntry ) {
            this.vdbModelEntry = vdbModelEntry;
        }

        @Override
        public void run() {
            // Get available servers and launch SelectTranslatorDialog
            // vdbModelEntry should not be null and should be a Physical model only

            if (vdbModelEntry != null) {
                String transName = vdbModelEntry.getSourceInfo().getSource(0).getTranslatorName();

                SelectTranslatorDialog dialog = new SelectTranslatorDialog(Display.getCurrent().getActiveShell());

                ITeiidTranslator initialSelection = null;
                ITeiidServer defServer = getDefaultServer();
                if (defServer != null && defServer.isConnected()) {
                    try {
                        initialSelection = defServer.getTranslator(transName);
                    } catch (Exception e) {
                        UTIL.log(IStatus.ERROR,
                                 e,
                                 UTIL.getString("VdbSourceConnectionHandler.Error_could_not_find_translator_for_name", transName)); //$NON-NLS-1$
                    }
                    dialog.setInitialSelection(initialSelection);
                }

                dialog.open();

                if (dialog.getReturnCode() == Window.OK) {
                    Object result = dialog.getFirstResult();
                    if (result != null && result instanceof ITeiidTranslator) {
                        vdbModelEntry.setTranslatorName(0, ((ITeiidTranslator)result).getName());
                    }
                }
            }
        }
    }

    /**
     * {@inheritDoc}
     * 
     * @see org.teiid.designer.vdb.connections.SourceHandler#getDataSourceNames()
     */
    @Override
    public String[] getDataSourceJndiNames() {
        ITeiidServer defaultServer = getDefaultServer();

        if ((defaultServer != null) && defaultServer.isConnected()) {
            Collection<ITeiidDataSource> dataSources = null;

            try {
                dataSources = defaultServer.getDataSources();
            } catch (Exception e) {
                UTIL.log(IStatus.ERROR,
                         e,
                         UTIL.getString("VdbSourceConnectionHandler.errorObtainingDataSources", defaultServer.getHost())); //$NON-NLS-1$
            }

            if (dataSources != null) {
                Collection<String> dataSourceJndiNames = new ArrayList<String>();

                for (ITeiidDataSource dataSource : dataSources) {
                    if (!dataSource.isPreview()) {
                        String sourceJndiName = dataSource.getPropertyValue(JNDI_PROPERTY_KEY);
                        if(CoreStringUtil.isEmpty(sourceJndiName)) {
                        	sourceJndiName=dataSource.getName();
                        }
                        dataSourceJndiNames.add(sourceJndiName);
                    }
                }

                return dataSourceJndiNames.toArray(new String[dataSourceJndiNames.size()]);
            }
        }

        return null;
    }

    ITeiidServer getDefaultServer() {
        return DqpPlugin.getInstance().getServerManager().getDefaultServer();
    }

    /**
     * {@inheritDoc}
     * 
     * @see org.teiid.designer.vdb.connections.SourceHandler#getTranslatorDefinitions(java.lang.String)
     */
    @Override
    public PropertyDefinition[] getTranslatorDefinitions( String translatorName ) {
    	return TranslatorUtils.getTranslatorPropertyDefinitions(translatorName, ITeiidTranslator.TranslatorPropertyType.OVERRIDE);
    }

    /**
     * {@inheritDoc}
     * 
     * @see org.teiid.designer.vdb.connections.SourceHandler#getTranslatorTypes()
     */
    @Override
    public String[] getTranslatorTypes() {
        ITeiidServer defaultServer = getDefaultServer();

        if ((defaultServer != null) && defaultServer.isConnected()) {
            Collection<ITeiidTranslator> translators = null;

            try {
                translators = defaultServer.getTranslators();
            } catch (Exception e) {
                UTIL.log(IStatus.ERROR,
                         e,
                         UTIL.getString("VdbSourceConnectionHandler.errorObtainingTranslators", defaultServer.getHost())); //$NON-NLS-1$
            }

            if (translators != null) {
                Collection<String> translatorTypes = new ArrayList<String>();

                for (ITeiidTranslator translator : translators) {
                    translatorTypes.add(translator.getName());
                }

                return translatorTypes.toArray(new String[translatorTypes.size()]);
            }
        }

        return null;
    }

    class TranslatorProperty implements PropertyDefinition {

        private final String className;

        boolean advanced;
        String[] allowedValues;
        String defaultValue;
        String description;
        String displayName;
        String id;
        boolean masked;
        boolean modifiable;
        boolean required;

        public TranslatorProperty( String className ) {
            this.className = className;
        }

        /**
         * {@inheritDoc}
         * 
         * @see java.lang.Object#equals(java.lang.Object)
         */
        @Override
        public boolean equals( Object obj ) {
            if (this == obj) {
                return true;
            }

            if ((obj == null) || (getClass() != obj.getClass())) {
                return false;
            }

            return this.id.equals(((TranslatorProperty)obj).id);
        }

        /**
         * {@inheritDoc}
         * 
         * PropertyDefinition#getAllowedValues()
         */
        @Override
        public String[] getAllowedValues() {
            return this.allowedValues;
        }

        /**
         * {@inheritDoc}
         * 
         * PropertyDefinition#getDefaultValue()
         */
        @Override
        public String getDefaultValue() {
            return this.defaultValue;
        }

        /**
         * {@inheritDoc}
         * 
         * PropertyDefinition#getDescription()
         */
        @Override
        public String getDescription() {
            return this.description;
        }

        /**
         * {@inheritDoc}
         * 
         * PropertyDefinition#getDisplayName()
         */
        @Override
        public String getDisplayName() {
            return this.displayName;
        }

        /**
         * {@inheritDoc}
         * 
         * PropertyDefinition#getId()
         */
        @Override
        public String getId() {
            return this.id;
        }

        /**
         * {@inheritDoc}
         * 
         * @see java.lang.Object#hashCode()
         */
        @Override
        public int hashCode() {
            return this.id.hashCode();
        }

        /**
         * {@inheritDoc}
         * 
         * PropertyDefinition#isAdvanced()
         */
        @Override
        public boolean isAdvanced() {
            return this.advanced;
        }

        /**
         * {@inheritDoc}
         * 
         * PropertyDefinition#isMasked()
         */
        @Override
        public boolean isMasked() {
            return this.masked;
        }

        /**
         * {@inheritDoc}
         * 
         * PropertyDefinition#isModifiable()
         */
        @Override
        public boolean isModifiable() {
            return this.modifiable;
        }

        /**
         * {@inheritDoc}
         * 
         * PropertyDefinition#isRequired()
         */
        @Override
        public boolean isRequired() {
            return this.required;
        }

        /**
         * {@inheritDoc}
         * 
         * PropertyDefinition#isValidValue(java.lang.String)
         */
        @Override
        public String isValidValue( String newValue ) {
            // if empty must have a value or a default value if required
            if (StringUtilities.isEmpty(newValue)) {
                // invalid if required and no default value
                if (isRequired() && StringUtilities.isEmpty(getDefaultValue())) {
                    return Util.getString("invalidNullPropertyValue", getDisplayName()); //$NON-NLS-1$
                }

                // OK to be null/empty
                return null;
            }

            if (Boolean.class.getName().equals(this.className) || Boolean.TYPE.getName().equals(this.className)) {
                if (!newValue.equalsIgnoreCase(Boolean.TRUE.toString()) && !newValue.equalsIgnoreCase(Boolean.FALSE.toString())) {
                    return Util.getString("invalidPropertyValueForType", newValue, Boolean.TYPE.getName()); //$NON-NLS-1$
                }
            } else if (Character.class.getName().equals(this.className) || Character.TYPE.getName().equals(this.className)) {
                if (newValue.length() != 1) {
                    return Util.getString("invalidPropertyValueForType", newValue, Character.TYPE.getName()); //$NON-NLS-1$
                }
            } else if (Byte.class.getName().equals(this.className) || Byte.TYPE.getName().equals(this.className)) {
                try {
                    Byte.parseByte(newValue);
                } catch (Exception e) {
                    return Util.getString("invalidPropertyValueForType", newValue, Byte.TYPE.getName()); //$NON-NLS-1$
                }
            } else if (Short.class.getName().equals(this.className) || Short.TYPE.getName().equals(this.className)) {
                try {
                    Short.parseShort(newValue);
                } catch (Exception e) {
                    return Util.getString("invalidPropertyValueForType", newValue, Short.TYPE.getName()); //$NON-NLS-1$
                }
            } else if (Integer.class.getName().equals(this.className) || Integer.TYPE.getName().equals(this.className)) {
                try {
                    Integer.parseInt(newValue);
                } catch (Exception e) {
                    return Util.getString("invalidPropertyValueForType", newValue, Integer.TYPE.getName()); //$NON-NLS-1$
                }
            } else if (Long.class.getName().equals(this.className) || Long.TYPE.getName().equals(this.className)) {
                try {
                    Long.parseLong(newValue);
                } catch (Exception e) {
                    return Util.getString("invalidPropertyValueForType", newValue, Long.TYPE.getName()); //$NON-NLS-1$
                }
            } else if (Float.class.getName().equals(this.className) || Float.TYPE.getName().equals(this.className)) {
                try {
                    Float.parseFloat(newValue);
                } catch (Exception e) {
                    return Util.getString("invalidPropertyValueForType", newValue, Float.TYPE.getName()); //$NON-NLS-1$
                }
            } else if (Double.class.getName().equals(this.className) || Double.TYPE.getName().equals(this.className)) {
                try {
                    Double.parseDouble(newValue);
                } catch (Exception e) {
                    return Util.getString("invalidPropertyValueForType", newValue, Double.TYPE.getName()); //$NON-NLS-1$
                }
            } else if (!String.class.getName().equals(this.className)) {
                return Util.getString("unknownPropertyType", this.displayName, this.className); //$NON-NLS-1$
            }

            // valid value
            return null;
        }

    }
}
